<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ActivityModel;

class ActivitiesController extends BaseController
{
    protected $activityModel;

    public function __construct()
    {
        $this->activityModel = new ActivityModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $filter = [
            'tanggal_dari' => $this->request->getGet('tanggal_dari') ?? '',
            'tanggal_sampai' => $this->request->getGet('tanggal_sampai') ?? ''
        ];
        $sort = $this->request->getGet('sort') ?? 'tanggal';
        $order = $this->request->getGet('order') ?? 'DESC';
        $page = (int)($this->request->getGet('page') ?? 1);

        $builder = $this->activityModel->searchAndFilter($search, $filter, $sort, $order);
        $total = $builder->countAllResults(false);
        
        $activities = $builder->get(10, ($page - 1) * 10)->getResultArray();

        $pager = \Config\Services::pager();
        $pager->store('activities', $page, 10, $total);

        // Debug: Log pagination info
        log_message('debug', "Activities Pagination - Total: {$total}, Page: {$page}, Activities Count: " . count($activities));

        $data = [
            'title' => 'Kelola Aktivitas Harian',
            'activities' => $activities,
            'pager' => $pager,
            'total' => $total, // Add total to data
            'search' => $search,
            'filter' => $filter,
            'sort' => $sort,
            'order' => $order
        ];

        return view('admin/activities/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Aktivitas Harian'
        ];
        return view('admin/activities/create', $data);
    }

    public function store()
    {
        $rules = [
            'tanggal' => 'required',
            'jam' => 'required',
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'deskripsi' => 'permit_empty',
            'foto' => 'permit_empty|uploaded[foto]|max_size[foto,2048]|ext_in[foto,jpg,jpeg,png,gif]',
            'video' => 'permit_empty|uploaded[video]|max_size[video,10240]|ext_in[video,mp4,avi,mov]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileFoto = $this->request->getFile('foto');
        $fileVideo = $this->request->getFile('video');
        
        $fotoName = null;
        $videoName = null;

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            $fotoName = $fileFoto->getRandomName();
            // Create directory if not exists
            if (!is_dir(FCPATH . 'assets/uploads/activities/')) {
                mkdir(FCPATH . 'assets/uploads/activities/', 0755, true);
            }
            $fileFoto->move(FCPATH . 'assets/uploads/activities/', $fotoName);
        }

        if ($fileVideo && $fileVideo->isValid() && !$fileVideo->hasMoved()) {
            $videoName = $fileVideo->getRandomName();
            // Create directory if not exists
            if (!is_dir(FCPATH . 'assets/uploads/activities/')) {
                mkdir(FCPATH . 'assets/uploads/activities/', 0755, true);
            }
            $fileVideo->move(FCPATH . 'assets/uploads/activities/', $videoName);
        }

        $data = [
            'tanggal' => $this->request->getPost('tanggal'),
            'jam' => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'foto' => $fotoName,
            'video' => $videoName
        ];

        if ($this->activityModel->insert($data)) {
            return redirect()->to('admin/activities')->with('success', 'Aktivitas berhasil ditambahkan!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan aktivitas!');
        }
    }

    public function edit($id)
    {
        $activity = $this->activityModel->find($id);
        
        if (!$activity) {
            return redirect()->to('admin/activities')->with('error', 'Aktivitas tidak ditemukan!');
        }

        $data = [
            'title' => 'Edit Aktivitas Harian',
            'activity' => $activity
        ];
        return view('admin/activities/edit', $data);
    }

    public function update($id)
    {
        $activity = $this->activityModel->find($id);
        
        if (!$activity) {
            return redirect()->to('admin/activities')->with('error', 'Aktivitas tidak ditemukan!');
        }

        $rules = [
            'tanggal' => 'required',
            'jam' => 'required',
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'deskripsi' => 'permit_empty',
            'foto' => 'permit_empty|uploaded[foto]|max_size[foto,2048]|ext_in[foto,jpg,jpeg,png,gif]',
            'video' => 'permit_empty|uploaded[video]|max_size[video,10240]|ext_in[video,mp4,avi,mov]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileFoto = $this->request->getFile('foto');
        $fileVideo = $this->request->getFile('video');
        
        $data = [
            'tanggal' => $this->request->getPost('tanggal'),
            'jam' => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi' => $this->request->getPost('deskripsi')
        ];

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            // Delete old file if exists
            if ($activity['foto'] && file_exists(FCPATH . 'assets/uploads/activities/' . $activity['foto'])) {
                unlink(FCPATH . 'assets/uploads/activities/' . $activity['foto']);
            }
            $fotoName = $fileFoto->getRandomName();
            // Create directory if not exists
            if (!is_dir(FCPATH . 'assets/uploads/activities/')) {
                mkdir(FCPATH . 'assets/uploads/activities/', 0755, true);
            }
            $fileFoto->move(FCPATH . 'assets/uploads/activities/', $fotoName);
            $data['foto'] = $fotoName;
        }

        if ($fileVideo && $fileVideo->isValid() && !$fileVideo->hasMoved()) {
            // Delete old file if exists
            if ($activity['video'] && file_exists(FCPATH . 'assets/uploads/activities/' . $activity['video'])) {
                unlink(FCPATH . 'assets/uploads/activities/' . $activity['video']);
            }
            $videoName = $fileVideo->getRandomName();
            // Create directory if not exists
            if (!is_dir(FCPATH . 'assets/uploads/activities/')) {
                mkdir(FCPATH . 'assets/uploads/activities/', 0755, true);
            }
            $fileVideo->move(FCPATH . 'assets/uploads/activities/', $videoName);
            $data['video'] = $videoName;
        }

        if ($this->activityModel->update($id, $data)) {
            return redirect()->to('admin/activities')->with('success', 'Aktivitas berhasil diupdate!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate aktivitas!');
        }
    }

    public function delete($id)
    {
        $activity = $this->activityModel->find($id);
        
        if (!$activity) {
            return redirect()->to('admin/activities')->with('error', 'Aktivitas tidak ditemukan!');
        }

        if ($activity['foto'] && file_exists(FCPATH . 'assets/uploads/activities/' . $activity['foto'])) {
            unlink(FCPATH . 'assets/uploads/activities/' . $activity['foto']);
        }

        if ($activity['video'] && file_exists(FCPATH . 'assets/uploads/activities/' . $activity['video'])) {
            unlink(FCPATH . 'assets/uploads/activities/' . $activity['video']);
        }

        if ($this->activityModel->delete($id)) {
            return redirect()->to('admin/activities')->with('success', 'Aktivitas berhasil dihapus!');
        } else {
            return redirect()->to('admin/activities')->with('error', 'Gagal menghapus aktivitas!');
        }
    }
}

